import numpy as np
import pandas as pd
import matplotlib.pyplot as plt

def in_window(phases, center, halfwidth):
    d = np.abs(((phases - center + 0.5) % 1.0) - 0.5)
    return d <= halfwidth

def window_counts(phases, centers=(0.0, 0.5), eps_window=0.10):
    m0 = in_window(phases, centers[0], eps_window)
    m1 = in_window(phases, centers[1], eps_window)
    both = m0 & m1  # expect zero if windows are disjoint
    C0 = int(m0.sum())
    C1 = int(m1.sum())
    neutral = len(phases) - int((m0 | m1).sum())
    cra_ok = (both.sum() == 0)
    return C0, C1, neutral / len(phases), cra_ok

def visibility_from_counts(C0, C1):
    denom = C0 + C1
    if denom == 0:
        return np.nan
    return (C0 - C1) / denom

def char_estimator(eta):
    return float(np.mean(np.cos(2*np.pi*eta)))

def sinc_2piJ(J):
    if J == 0.0:
        return 1.0
    return float(np.sin(2*np.pi*J) / (2*np.pi*J))

def run_uniform_suite(K=50000, eps_window=0.10, centers=(0.0,0.5), base_seed=20251031):
    J_values = [0.0] + [round(x, 2) for x in np.arange(0.02, 0.25 + 1e-9, 0.02).tolist()]
    records = []
    for i, J in enumerate(J_values):
        local_rng = np.random.Generator(np.random.PCG64(base_seed + i*17))
        if J == 0.0:
            eta = np.zeros(K, dtype=float)
        else:
            eta = local_rng.uniform(-J, J, size=K)
        phases = (eta) % 1.0
        C0, C1, neutral_rate, cra_ok = window_counts(phases, centers, eps_window)
        vis = visibility_from_counts(C0, C1)
        vis_norm = char_estimator(eta)
        pred = sinc_2piJ(J)
        ratio = vis_norm / pred if pred != 0 else np.nan
        records.append({
            "J": J,
            "visibility": vis,
            "visibility_norm": vis_norm,
            "pred_sinc": pred,
            "ratio": ratio,
            "neutral_rate": neutral_rate,
            "cra_consistency_flag": bool(cra_ok),
        })
    return pd.DataFrame.from_records(records)

def run_gaussian_suite(K=50000, base_seed=20251031):
    sigma_values = [round(x, 2) for x in np.arange(0.02, 0.20 + 1e-9, 0.02).tolist()]
    records = []
    for i, sigma in enumerate(sigma_values):
        local_rng = np.random.Generator(np.random.PCG64(303 + i*31 + base_seed % 97))
        eta = local_rng.normal(loc=0.0, scale=sigma, size=K)
        vis_norm = char_estimator(eta)
        pred = float(np.exp(-2*(np.pi**2)*(sigma**2)))
        ratio = vis_norm / pred if pred != 0 else np.nan
        records.append({
            "sigma": sigma,
            "visibility_norm": vis_norm,
            "pred_exp": pred,
            "ratio": ratio,
        })
    return pd.DataFrame.from_records(records)

def run_twopoint_suite(K=50000, base_seed=20251031):
    a_values = [round(x, 2) for x in np.arange(0.02, 0.20 + 1e-9, 0.02).tolist()]
    records = []
    for i, a in enumerate(a_values):
        local_rng = np.random.Generator(np.random.PCG64(707 + i*43 + base_seed % 103))
        eta = np.where(local_rng.random(K) < 0.5, -a, a)
        vis_norm = char_estimator(eta)
        pred = float(np.cos(2*np.pi*a))
        ratio = vis_norm / pred if pred != 0 else np.nan
        records.append({
            "a": a,
            "visibility_norm": vis_norm,
            "pred_cos": pred,
            "ratio": ratio,
        })
    return pd.DataFrame.from_records(records)

def rmse(y_true, y_pred):
    arr = np.asarray(y_true) - np.asarray(y_pred)
    return float(np.sqrt(np.mean(arr*arr)))

def is_monotone_decreasing(arr, atol=1e-12):
    arr = np.asarray(arr, dtype=float)
    diffs = np.diff(arr)
    return bool(np.all(diffs <= atol))

def write_uniform_plot(df_uniform, path_png):
    plt.figure(figsize=(6,4))
    plt.plot(df_uniform["J"], df_uniform["visibility_norm"], marker="o", label="Measured (char-est.)")
    plt.plot(df_uniform["J"], df_uniform["pred_sinc"], marker="s", linestyle="--", label="Prediction sin(2πJ)/(2πJ)")
    plt.xlabel("J (cycles)")
    plt.ylabel("visibility_norm")
    plt.title("Uniform jitter: visibility_norm vs prediction")
    plt.legend()
    plt.tight_layout()
    plt.savefig(path_png)
    plt.close()
